/**
  ******************************************************************************
  * @file    main.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   Program entry point
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 

/* Includes ------------------------------------------------------------------*/
#include "stm32f4_discovery.h"

#define LED_ORANGE LED3
#define LED_GREEN  LED4
#define LED_BLUE   LED6
#define LED_RED    LED5

volatile unsigned long g_MSCtr;

// constants and code to (ab)use sys tick handler for free running timer copied from
// freertos

#define configCPU_CLOCK_HZ				( 168000000UL )	
#define configTICK_RATE_HZ				( 1000 )

#define configSYSTICK_CLOCK_HZ configCPU_CLOCK_HZ
/* Ensure the SysTick is clocked at the same frequency as the core. */
#define portNVIC_SYSTICK_CLK_BIT	( 1UL << 2UL )

/* Constants required to manipulate the core.  Registers first... */
#define portNVIC_SYSTICK_CTRL_REG			( * ( ( volatile unsigned long * ) 0xe000e010 ) )
#define portNVIC_SYSTICK_LOAD_REG			( * ( ( volatile unsigned long * ) 0xe000e014 ) )
#define portNVIC_SYSTICK_CURRENT_VALUE_REG	( * ( ( volatile unsigned long * ) 0xe000e018 ) )
#define portNVIC_SYSPRI2_REG				( * ( ( volatile unsigned long * ) 0xe000ed20 ) )
/* ...then bits in the registers. */
#define portNVIC_SYSTICK_INT_BIT			( 1UL << 1UL )
#define portNVIC_SYSTICK_ENABLE_BIT			( 1UL << 0UL )
#define portNVIC_SYSTICK_COUNT_FLAG_BIT		( 1UL << 16UL )
#define portNVIC_PENDSVCLEAR_BIT 			( 1UL << 27UL )
#define portNVIC_PEND_SYSTICK_CLEAR_BIT		( 1UL << 25UL )

/** @brief sets up the system timer to interrupt once every millisecond. Sets Interrupt Priority to highest. 
 *
 *
 *  @return none
 */

void SetupCortexSysTickTimer( void )
{
	/* Configure SysTick to interrupt at the requested rate. */
	portNVIC_SYSTICK_LOAD_REG = ( configSYSTICK_CLOCK_HZ / configTICK_RATE_HZ ) - 1UL;
	portNVIC_SYSTICK_CTRL_REG = ( portNVIC_SYSTICK_CLK_BIT | portNVIC_SYSTICK_INT_BIT | portNVIC_SYSTICK_ENABLE_BIT );
    NVIC_SetPriority (SysTick_IRQn, (1<<__NVIC_PRIO_BITS) - 1);  /* set Priority for Cortex-M0 System Interrupts */
}

/** @brief Sys Tick Interrupt Handler
 *
 *
 *  @return none
 */

void SysTick_Handler(void)
{
    g_MSCtr++; 
}

/** @brief Delays the caller for the amount of milliseconds specified in the argument
 *
 *  Implements a busy waiting delay
 *
 *  @param p_Time Number of Milliseconds to delay
 *  @return none
 */

void Delay(__IO unsigned long p_Time)
{ 
    unsigned long a_Before = g_MSCtr;
    unsigned long a_After = a_Before + p_Time;
    while (g_MSCtr < a_After){};
}

#define BLINKEDELAYINMS 1000

/** @brief Main function body.
 *
 *
 *  @return none (executes infinite loop)
 */

void BlinkTest(void)
{
  while (1)
  {

    STM_EVAL_LEDToggle(LED_ORANGE);
    STM_EVAL_LEDToggle(LED_BLUE);

    Delay(BLINKEDELAYINMS);

    STM_EVAL_LEDToggle(LED_GREEN);
    STM_EVAL_LEDToggle(LED_RED);

    Delay(BLINKEDELAYINMS);
  }  
}

/** @brief Application entry point
 *
 *
 *  @return Dummy (required by unused C Runtime Code). Function does not return.
 */

int main(void)
{
  
  /* Initialize Leds mounted on STM32F4-Discovery board */
  STM_EVAL_LEDInit(LED_GREEN);
  STM_EVAL_LEDInit(LED_ORANGE);
  STM_EVAL_LEDInit(LED_RED);
  STM_EVAL_LEDInit(LED_BLUE);

  SetupCortexSysTickTimer();
	__asm volatile(" cpsie i \n" /* Globally enable interrupts. */);

  BlinkTest();
}
